#!/usr/bin/env python3
"""
Generate interactive HTML LLM cost comparison reports.

Usage:
    python generate_comparison.py --output comparison.html
    python generate_comparison.py -o report.html --title "My Cost Analysis"
"""

import argparse
import json
from datetime import datetime

# Import pricing from calculator
from cost_calculator import PRICING


def generate_html_report(
    title: str = "LLM Cost Comparison Calculator",
    output_file: str = "llm-comparison.html"
) -> str:
    """Generate an interactive HTML cost comparison report."""
    
    # Convert pricing to JSON for JavaScript
    pricing_json = json.dumps(PRICING)
    
    html = f'''<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{title}</title>
    <style>
        :root {{
            --primary: #2563eb;
            --primary-dark: #1d4ed8;
            --success: #16a34a;
            --warning: #d97706;
            --danger: #dc2626;
            --bg: #f8fafc;
            --card: #ffffff;
            --border: #e2e8f0;
            --text: #1e293b;
            --text-muted: #64748b;
        }}
        
        * {{ box-sizing: border-box; margin: 0; padding: 0; }}
        
        body {{
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: var(--bg);
            color: var(--text);
            line-height: 1.6;
            padding: 2rem;
        }}
        
        .container {{
            max-width: 1400px;
            margin: 0 auto;
        }}
        
        h1 {{
            font-size: 1.875rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }}
        
        .subtitle {{
            color: var(--text-muted);
            margin-bottom: 2rem;
        }}
        
        .controls {{
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 1.5rem;
            margin-bottom: 1.5rem;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }}
        
        .control-group {{
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }}
        
        label {{
            font-weight: 500;
            font-size: 0.875rem;
            color: var(--text-muted);
        }}
        
        input[type="number"], input[type="text"], select {{
            padding: 0.75rem;
            border: 1px solid var(--border);
            border-radius: 8px;
            font-size: 1rem;
            width: 100%;
        }}
        
        input:focus, select:focus {{
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(37, 99, 235, 0.1);
        }}
        
        .btn {{
            background: var(--primary);
            color: white;
            border: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            cursor: pointer;
            font-weight: 500;
            transition: background 0.2s;
        }}
        
        .btn:hover {{
            background: var(--primary-dark);
        }}
        
        .btn-secondary {{
            background: var(--card);
            color: var(--text);
            border: 1px solid var(--border);
        }}
        
        .btn-secondary:hover {{
            background: var(--bg);
        }}
        
        .results {{
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
            overflow: hidden;
        }}
        
        table {{
            width: 100%;
            border-collapse: collapse;
        }}
        
        th, td {{
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid var(--border);
        }}
        
        th {{
            background: var(--bg);
            font-weight: 600;
            font-size: 0.875rem;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            color: var(--text-muted);
        }}
        
        tr:hover {{
            background: var(--bg);
        }}
        
        .cost-cell {{
            font-family: 'SF Mono', Monaco, monospace;
            font-weight: 500;
        }}
        
        .cost-low {{ color: var(--success); }}
        .cost-mid {{ color: var(--warning); }}
        .cost-high {{ color: var(--danger); }}
        
        .model-badge {{
            display: inline-block;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 500;
            margin-left: 0.5rem;
        }}
        
        .badge-anthropic {{ background: #fef3c7; color: #92400e; }}
        .badge-openai {{ background: #dcfce7; color: #166534; }}
        .badge-google {{ background: #dbeafe; color: #1e40af; }}
        .badge-mistral {{ background: #fce7f3; color: #9d174d; }}
        .badge-meta {{ background: #e0e7ff; color: #3730a3; }}
        .badge-xai {{ background: #f3e8ff; color: #6b21a8; }}
        .badge-deepseek {{ background: #cffafe; color: #0e7490; }}
        .badge-amazon {{ background: #ffedd5; color: #c2410c; }}
        
        .share-url {{
            background: var(--bg);
            padding: 1rem;
            border-radius: 8px;
            margin-top: 1rem;
            display: flex;
            gap: 0.5rem;
        }}
        
        .share-url input {{
            flex: 1;
            font-family: monospace;
            font-size: 0.875rem;
        }}
        
        .summary {{
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin-bottom: 1.5rem;
        }}
        
        .summary-card {{
            background: var(--card);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 1rem;
            text-align: center;
        }}
        
        .summary-value {{
            font-size: 1.5rem;
            font-weight: 700;
            color: var(--primary);
        }}
        
        .summary-label {{
            font-size: 0.875rem;
            color: var(--text-muted);
        }}
        
        .checkbox-group {{
            display: flex;
            flex-wrap: wrap;
            gap: 0.5rem;
        }}
        
        .checkbox-item {{
            display: flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.5rem;
            background: var(--bg);
            border-radius: 4px;
            font-size: 0.875rem;
        }}
        
        @media (max-width: 768px) {{
            body {{ padding: 1rem; }}
            .controls {{ grid-template-columns: 1fr; }}
        }}
    </style>
</head>
<body>
    <div class="container">
        <h1>🤖 {title}</h1>
        <p class="subtitle">Compare API costs across LLM providers • Generated {datetime.now().strftime("%Y-%m-%d")}</p>
        
        <div class="controls">
            <div class="control-group">
                <label for="inputTokens">Input Tokens</label>
                <input type="number" id="inputTokens" value="1000000" min="0" step="1000">
            </div>
            <div class="control-group">
                <label for="outputTokens">Output Tokens</label>
                <input type="number" id="outputTokens" value="100000" min="0" step="1000">
            </div>
            <div class="control-group">
                <label for="useCached">Use Cached Pricing</label>
                <select id="useCached">
                    <option value="false">No (Standard)</option>
                    <option value="true">Yes (Cached Input)</option>
                </select>
            </div>
            <div class="control-group">
                <label>&nbsp;</label>
                <div style="display: flex; gap: 0.5rem;">
                    <button class="btn" onclick="calculate()">Calculate</button>
                    <button class="btn btn-secondary" onclick="shareUrl()">Share URL</button>
                </div>
            </div>
        </div>
        
        <div class="summary" id="summary"></div>
        
        <div class="results">
            <table>
                <thead>
                    <tr>
                        <th>Model</th>
                        <th>Input $/1M</th>
                        <th>Output $/1M</th>
                        <th>Input Cost</th>
                        <th>Output Cost</th>
                        <th>Total Cost</th>
                    </tr>
                </thead>
                <tbody id="resultsBody"></tbody>
            </table>
        </div>
        
        <div class="share-url" id="shareSection" style="display: none;">
            <input type="text" id="shareUrlInput" readonly>
            <button class="btn btn-secondary" onclick="copyUrl()">Copy</button>
        </div>
    </div>
    
    <script>
        const PRICING = {pricing_json};
        
        function getProvider(model) {{
            if (model.startsWith('claude')) return 'anthropic';
            if (model.startsWith('gpt') || model.startsWith('o1') || model.startsWith('o3')) return 'openai';
            if (model.startsWith('gemini')) return 'google';
            if (model.startsWith('mistral') || model.startsWith('codestral') || model.startsWith('ministral') || model.startsWith('pixtral')) return 'mistral';
            if (model.startsWith('llama')) return 'meta';
            if (model.startsWith('grok')) return 'xai';
            if (model.startsWith('deepseek')) return 'deepseek';
            if (model.startsWith('amazon')) return 'amazon';
            return 'other';
        }}
        
        function formatCost(cost) {{
            if (cost < 0.01) return '$' + cost.toFixed(4);
            if (cost < 1) return '$' + cost.toFixed(3);
            if (cost < 100) return '$' + cost.toFixed(2);
            return '$' + cost.toFixed(0);
        }}
        
        function getCostClass(cost, allCosts) {{
            const sorted = [...allCosts].sort((a, b) => a - b);
            const idx = sorted.indexOf(cost);
            const pct = idx / sorted.length;
            if (pct < 0.33) return 'cost-low';
            if (pct < 0.67) return 'cost-mid';
            return 'cost-high';
        }}
        
        function calculate() {{
            const inputTokens = parseInt(document.getElementById('inputTokens').value) || 0;
            const outputTokens = parseInt(document.getElementById('outputTokens').value) || 0;
            const useCached = document.getElementById('useCached').value === 'true';
            
            const results = [];
            
            for (const [model, prices] of Object.entries(PRICING)) {{
                const [inputPrice, outputPrice, cachedPrice] = prices;
                const effectiveInput = useCached && cachedPrice ? cachedPrice : inputPrice;
                
                const inputCost = (inputTokens / 1000000) * effectiveInput;
                const outputCost = (outputTokens / 1000000) * outputPrice;
                const totalCost = inputCost + outputCost;
                
                results.push({{
                    model,
                    provider: getProvider(model),
                    inputPrice: effectiveInput,
                    outputPrice,
                    inputCost,
                    outputCost,
                    totalCost
                }});
            }}
            
            results.sort((a, b) => a.totalCost - b.totalCost);
            
            const allTotals = results.map(r => r.totalCost);
            
            // Summary
            const summaryHtml = `
                <div class="summary-card">
                    <div class="summary-value">${{results.length}}</div>
                    <div class="summary-label">Models Compared</div>
                </div>
                <div class="summary-card">
                    <div class="summary-value">${{formatCost(results[0].totalCost)}}</div>
                    <div class="summary-label">Lowest Cost</div>
                </div>
                <div class="summary-card">
                    <div class="summary-value">${{formatCost(results[results.length-1].totalCost)}}</div>
                    <div class="summary-label">Highest Cost</div>
                </div>
                <div class="summary-card">
                    <div class="summary-value">${{(results[results.length-1].totalCost / results[0].totalCost).toFixed(1)}}x</div>
                    <div class="summary-label">Price Range</div>
                </div>
            `;
            document.getElementById('summary').innerHTML = summaryHtml;
            
            // Table
            let html = '';
            for (const r of results) {{
                const costClass = getCostClass(r.totalCost, allTotals);
                html += `
                    <tr>
                        <td>
                            ${{r.model}}
                            <span class="model-badge badge-${{r.provider}}">${{r.provider}}</span>
                        </td>
                        <td>${{formatCost(r.inputPrice)}}</td>
                        <td>${{formatCost(r.outputPrice)}}</td>
                        <td class="cost-cell">${{formatCost(r.inputCost)}}</td>
                        <td class="cost-cell">${{formatCost(r.outputCost)}}</td>
                        <td class="cost-cell ${{costClass}}">${{formatCost(r.totalCost)}}</td>
                    </tr>
                `;
            }}
            document.getElementById('resultsBody').innerHTML = html;
        }}
        
        function shareUrl() {{
            const inputTokens = document.getElementById('inputTokens').value;
            const outputTokens = document.getElementById('outputTokens').value;
            const useCached = document.getElementById('useCached').value;
            
            const params = `it=${{inputTokens}}&ot=${{outputTokens}}&cached=${{useCached}}`;
            const url = window.location.href.split('#')[0] + '#' + params;
            
            document.getElementById('shareUrlInput').value = url;
            document.getElementById('shareSection').style.display = 'flex';
        }}
        
        function copyUrl() {{
            const input = document.getElementById('shareUrlInput');
            input.select();
            document.execCommand('copy');
            alert('URL copied to clipboard!');
        }}
        
        function loadFromUrl() {{
            const hash = window.location.hash.substring(1);
            if (!hash) return;
            
            const params = new URLSearchParams(hash);
            if (params.has('it')) document.getElementById('inputTokens').value = params.get('it');
            if (params.has('ot')) document.getElementById('outputTokens').value = params.get('ot');
            if (params.has('cached')) document.getElementById('useCached').value = params.get('cached');
        }}
        
        // Initialize
        loadFromUrl();
        calculate();
    </script>
</body>
</html>'''
    
    with open(output_file, 'w') as f:
        f.write(html)
    
    return output_file


def main():
    parser = argparse.ArgumentParser(description="Generate LLM cost comparison HTML")
    parser.add_argument("-o", "--output", type=str, default="llm-comparison.html",
                        help="Output HTML file path")
    parser.add_argument("-t", "--title", type=str, default="LLM Cost Comparison Calculator",
                        help="Page title")
    
    args = parser.parse_args()
    
    output = generate_html_report(args.title, args.output)
    print(f"Generated: {output}")


if __name__ == "__main__":
    main()
