---
name: uriburner-sparql-agent
description: "Semantic data discovery agent for SPARQL, RDF graphs, entity exploration, and knowledge graph queries. Implements KG-first workflow for fact lookup, citation verification, and graph exploration through URIBurner MCP tools."
license: "MIT"
---

# URIBurner SPARQL Agent

## Overview

This workflow enables semantic data discovery and query execution using a knowledge-graph-first protocol. The agent follows a structured process: attempt label matching → semantic breakdown → entity exploration → query construction with IRI-backed results.

## Quick Start

### Basic Label Match Query

For prompt "Show me questions about climate change":

```sparql
SELECT DISTINCT ?entity ?value
WHERE {
  ?entity ?attribute ?value.
  ?value bif:contains("'climate change'") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT 10
```

Execute using: `execute_spasql_query` or `sparqlRemoteQuery`

### Entity Type Exploration

Discover what entity types are available in the graph:

```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX schema: <http://schema.org/>
PREFIX skos: <http://www.w3.org/2004/02/skos/core#>

SELECT ?type (COUNT(?entity) AS ?count) (SAMPLE(?entity) AS ?sampleEntity)
WHERE {
  ?entity rdf:type ?type .
  FILTER (?type IN (
    schema:FAQPage,
    schema:Question,
    schema:DefinedTermSet,
    skos:ConceptScheme,
    schema:HowTo
  ))
}
GROUP BY ?type
ORDER BY DESC(?count)
```

## Workflow: KG-First Protocol

Follow these steps in order:

### Step 1: Basic Label Match
Use the label-match template with `bif:contains()` (Virtuoso preference) to match prompt text against indexed labels. Set result limit (default: 10, configurable via `/limit` command).

**If found:** Return results with hyperlinked IRIs. Skip to Step 5.

**If not found:** Try up to 3 semantically similar prompt variants, then proceed to Step 2.

### Step 2: Semantic Breakdown
Decompose the prompt into subject, predicate, and object components. Determine relevant entity types (e.g., `schema:Question`, `schema:DefinedTermSet`, `skos:ConceptScheme`, `schema:HowTo`).

### Step 3: Explore & Identify
Identify which graph resources to query using entity exploration queries. Run a single combined query per target processor that retrieves:
- Textual answers
- Source IRIs for relevant entities
- Optional ranking/score for semantic similarity

### Step 4: Query for IRIs and Text
Execute constructed query against target endpoint. If no results on primary endpoint, retry against fallback endpoints in order:
1. `https://linkeddata.uriburner.com/sparql` (default local)
2. `https://kingsley.idehen.net/sparql`
3. `https://demo.openlinksw.com/sparql`

### Step 5: Format with Encoded Hyperlinks
Present results in Markdown with hyperlinks to `/describe/?uri={encoded_IRI}` service. **Critical:** URL-encode all IRIs. Domain must match result origin:
- Local: `https://linkeddata.uriburner.com/describe/?uri={encoded_IRI}`
- Remote: `https://kingsley.idehen.net/describe/?uri={encoded_IRI}`

**Example:** Entity IRI `https://kingsley.idehen.net/DAV/home/file.pdf#Question_1` becomes:
```
https://kingsley.idehen.net/describe/?uri=https%3A%2F%2Fkingsley.idehen.net%2FDAV%2Fhome%2Ffile.pdf%23Question_1
```

## Available Tools

| Tool | Use Case |
|------|----------|
| `execute_spasql_query` | Execute SPARQL-within-SQL queries on local instance |
| `sparqlRemoteQuery` | Query remote SPARQL endpoints with explicit endpoint URL |
| `EXECUTE_SQL_SCRIPT` | Execute SQL scripts with multiple statements |
| `WEB_FETCH` | Retrieve and process web content for sponging/enrichment |
| `getLatestNews` | Fetch latest indexed news articles |

## Query Processing Rules

### For Virtuoso Backend (Default)
- Prefer `bif:contains()` for full-text search over `REGEX()`
- Combine multiple terms with logical operators (AND, OR) in single clause
- Use pragma for equivalent-class reasoning:
  ```
  DEFINE input:inference 'urn:owl:equivalent:class:inference:rules'
  ```

### For Non-Virtuoso Backends
- Use standard SPARQL `REGEX()` functions
- Combine filters with `&&`, `||` operators

## Commands

| Command | Effect |
|---------|--------|
| `/help` | Show common issues and usage help |
| `/query` | Assist with formulating SPARQL-within-SQL queries |
| `/config` | Configure driver and connection settings |
| `/troubleshoot` | Diagnose connection or driver issues |
| `/limit N` | Set result limit for queries (default: 10) |
| `/kg-on` | Force KG workflow for current and subsequent prompts |
| `/kg-off` | Disable KG workflow until re-enabled |
| `/kg-verify` | Run KG workflow, then synthesize summary with marked provenance |

## Fallback Strategy

If KG workflow initial steps fail:

1. Retry exploratory query with reasoning/inference enabled
2. Try additional remote endpoints in configured order
3. Run semantic similarity-based refinement and retry
4. If all attempts fail, report to user with:
   - Executed queries
   - Endpoints attempted
   - Reason(s) for failure
   - Option to synthesize without KG or provide semantic variants

**Important:** Finding a result on a remote endpoint does NOT change the default data-source choice for subsequent prompts.

## Operational Constraints

- **Max queries per prompt:** 6
- **Per-query timeout:** 30 seconds (configurable)
- **Semantic variant retries:** 2
- **Default result limit:** 10
- **Max parallel endpoint queries:** 2
- **Maximum total KG time:** 120 seconds

## Enforcement and Audit

When KG workflow is required or fails, provide audit block with:
- `query_text`
- `endpoint_url`
- `start_timestamp` / `end_timestamp`
- `status_code` / error details
- `result_summary`

If strict mode (`kg_mandatory_for_all_prompts = true`) is enabled, the agent MUST complete KG workflow or exit with audit block.

## Reference

For detailed query templates and advanced patterns, see `/references/predefined-queries.md`.
