# Real-World Query Examples

## Table of Contents
1. [FAQ & Question Discovery](#faq--question-discovery)
2. [Ontology & Schema Queries](#ontology--schema-queries)
3. [Knowledge Graph Exploration](#knowledge-graph-exploration)
4. [Citation & Reference Lookup](#citation--reference-lookup)
5. [Multi-Step Workflows](#multi-step-workflows)

## FAQ & Question Discovery

### Example 1: Find Questions About a Topic
**User Query:** "What questions exist about renewable energy?"

**Step 1: Label Match**
```sparql
SELECT DISTINCT ?question ?text ?sc
WHERE {
  ?question rdf:type schema:Question .
  ?question schema:text ?text .
  ?text bif:contains("'renewable energy'") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT 10
```

**Expected Result:**
| question | text | sc |
|----------|------|-----|
| `https://linkeddata.uriburner.com/DAV/home/faq.rdf#Q_renewable_1` | How does renewable energy reduce carbon emissions? | 95.2 |
| `https://linkeddata.uriburner.com/DAV/home/faq.rdf#Q_renewable_2` | What are the benefits of renewable energy? | 87.5 |

**Step 5: Format with Hyperlinks**
- [How does renewable energy reduce carbon emissions?](https://linkeddata.uriburner.com/describe/?uri=https%3A%2F%2Flinkeddata.uriburner.com%2FDAV%2Fhome%2Ffaq.rdf%23Q_renewable_1)
- [What are the benefits of renewable energy?](https://linkeddata.uriburner.com/describe/?uri=https%3A%2F%2Flinkeddata.uriburner.com%2FDAV%2Fhome%2Ffaq.rdf%23Q_renewable_2)

---

### Example 2: Find FAQ Pages by Category
**User Query:** "Show me FAQ pages about machine learning"

**Workflow:**

```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX schema: <http://schema.org/>

SELECT ?faqPage ?name ?description ?url
WHERE {
  ?faqPage rdf:type schema:FAQPage .
  ?faqPage schema:name ?name .
  OPTIONAL { ?faqPage schema:description ?description . }
  OPTIONAL { ?faqPage schema:url ?url . }
  ?name bif:contains("'machine learning'") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT 10
```

**Result Presentation:**
Each FAQ Page IRI gets linked to `/describe/?uri={encoded_IRI}` for detailed exploration.

---

## Ontology & Schema Queries

### Example 3: Discover Ontology Structure
**User Query:** "What ontologies are available in this graph?"

**Execution:**
```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX owl: <http://www.w3.org/2002/07/owl#>
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>

SELECT ?ontology ?label (COUNT(?class) AS ?classCount)
WHERE {
  ?ontology rdf:type owl:Ontology .
  ?ontology rdfs:label ?label .
  ?class rdfs:isDefinedBy ?ontology .
  ?class rdf:type rdfs:Class .
}
GROUP BY ?ontology ?label
ORDER BY DESC(?classCount)
LIMIT 20
```

**Use Case:** Understand available data models before constructing queries.

---

### Example 4: Map Entity Types to Data Sources
**User Query:** "Which namespaces have Question entities?"

**Execution:**
```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX schema: <http://schema.org/>

SELECT ?namespace (COUNT(?question) AS ?questionCount)
WHERE {
  ?question rdf:type schema:Question .
  BIND(STRBEFORE(STR(?question), "#") AS ?ns)
  BIND(STRBEFORE(?ns, "/") AS ?namespace)
}
GROUP BY ?namespace
ORDER BY DESC(?questionCount)
```

**Output:** Helps route queries to appropriate endpoints.

---

## Knowledge Graph Exploration

### Example 5: Full Knowledge Graph Inventory
**User Query:** "Show me entity type distribution across the graph"

**Step 1: Entity Type Discovery**
```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX schema: <http://schema.org/>
PREFIX skos: <http://www.w3.org/2004/02/skos/core#>

SELECT ?type (COUNT(?entity) AS ?count) (SAMPLE(?entity) AS ?sample)
WHERE {
  ?entity rdf:type ?type .
  FILTER (?type IN (
    schema:FAQPage,
    schema:Question,
    schema:DefinedTermSet,
    skos:ConceptScheme,
    schema:HowTo
  ))
}
GROUP BY ?type
ORDER BY DESC(?count)
```

**Result:**
| type | count | sample |
|------|-------|--------|
| schema:Question | 1523 | `https://linkeddata.uriburner.com/DAV/home/faq.rdf#Q_1` |
| schema:FAQPage | 47 | `https://linkeddata.uriburner.com/DAV/home/faqs/main` |
| schema:HowTo | 234 | `https://linkeddata.uriburner.com/DAV/home/howto.rdf#HT_1` |

---

### Example 6: Semantic Relationships Discovery
**User Query:** "How are questions related to other entities?"

**Semantic Breakdown:**
- Subject: Questions
- Predicate: Links/relationships
- Object: Other entity types

**Query:**
```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX schema: <http://schema.org/>

SELECT ?predicate (COUNT(?question) AS ?relationCount)
WHERE {
  ?question rdf:type schema:Question .
  ?question ?predicate ?object .
  FILTER (!STRSTARTS(STR(?predicate), STR(rdf:)))
}
GROUP BY ?predicate
ORDER BY DESC(?relationCount)
LIMIT 15
```

**Use:** Identify common predicates for constructing more specific queries.

---

## Citation & Reference Lookup

### Example 7: Find Cited Sources
**User Query:** "What sources are cited for information about climate change?"

**Workflow:**
```sparql
PREFIX schema: <http://schema.org/>
PREFIX dct: <http://purl.org/dc/terms/>

SELECT ?entity ?cite ?source
WHERE {
  ?entity rdf:type schema:Question .
  ?entity schema:text ?text .
  ?text bif:contains("'climate change'") option (score ?sc) .
  OPTIONAL { ?entity dct:source ?cite . }
  OPTIONAL { ?cite schema:url ?source . }
}
ORDER BY DESC(?sc)
LIMIT 10
```

---

### Example 8: Verify Citation Consistency
**User Query:** "Verify that question Q_1 consistently cites its sources"

**KG Workflow:**
```sparql
PREFIX schema: <http://schema.org/>
PREFIX dct: <http://purl.org/dc/terms/>

SELECT ?question ?citedSource ?isValid
WHERE {
  ?question schema:identifier "Q_1" .
  OPTIONAL { ?question dct:source ?citedSource . }
  OPTIONAL { ?citedSource schema:url ?url . }
  BIND(BOUND(?url) AS ?isValid)
}
```

**Result Interpretation:**
- `?isValid = true` → Citation has valid URL
- `?isValid = false` → Citation missing or incomplete

---

## Multi-Step Workflows

### Example 9: Complex Discovery: Find "How-To" Guides About a Technology

**User Query:** "Show me how-to guides about implementing OAuth"

**Step 1: Label Match**
```sparql
PREFIX schema: <http://schema.org/>

SELECT DISTINCT ?howto ?name ?sc
WHERE {
  ?howto rdf:type schema:HowTo .
  ?howto schema:name ?name .
  ?name bif:contains("'OAuth' OR 'authentication'") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT 10
```

**Step 2: If no direct match, Semantic Breakdown**
Terms: "OAuth", "authentication", "implementation", "tutorial"

```sparql
PREFIX schema: <http://schema.org/>

SELECT ?howto ?name ?description ?step
WHERE {
  ?howto rdf:type schema:HowTo .
  ?howto schema:name ?name .
  ?howto schema:description ?description .
  OPTIONAL { ?howto schema:step ?step . }
  (
    (?description bif:contains("'OAuth'") option (score ?sc1)) ||
    (?step bif:contains("'authentication'") option (score ?sc2))
  )
  BIND(COALESCE(?sc1, ?sc2, 0) AS ?score)
}
ORDER BY DESC(?score)
LIMIT 10
```

**Step 5: Format Results**
Each result links to `/describe/?uri={encoded_IRI}` for full details.

---

### Example 10: Entity Enrichment Pipeline

**Scenario:** You found a Question entity and want to enrich it with related information.

**Query:**
```sparql
PREFIX schema: <http://schema.org/>
PREFIX dct: <http://purl.org/dc/terms/>
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>

SELECT ?question ?text ?answer ?relatedEntity ?dateModified
WHERE {
  ?question rdf:type schema:Question .
  ?question schema:identifier "{{question_id}}" .
  ?question schema:text ?text .
  OPTIONAL { ?question schema:acceptedAnswer ?answer . }
  OPTIONAL { ?question schema:relatedLink ?relatedEntity . }
  OPTIONAL { ?question dct:modified ?dateModified . }
}
```

**Output:** Complete question profile with answers, related entities, and metadata.

---

## Command-Line Usage Examples

### Using `/query` Command
```
/query Find FAQ pages about Virtuoso database

→ Agent constructs and executes label-match query
→ Returns results with IRI links
```

### Using `/limit` Command
```
/limit 20

→ Sets result limit to 20 for subsequent queries
→ Persists until changed
```

### Using `/kg-verify` Command
```
/kg-verify What is RDF?

→ Executes full KG workflow (all 5 steps)
→ Returns results with explicit provenance markers
→ Shows: [Query] → [Endpoint] → [Result IRI] → [Describe Link]
```

### Using `/kg-on` and `/kg-off`
```
/kg-on
  → Forces strict KG-first workflow for all queries

/query Show me ontologies
  → Executes full KG-first protocol

/kg-off
  → Returns to default behavior
```

---

## Performance Tips

1. **Use `bif:contains()` for Virtuoso** - Much faster than `REGEX()`
2. **Combine terms in single clause** - Avoid multiple `FILTER` blocks
3. **Limit result sets** - Use `/limit 10` for initial exploratory queries
4. **Add type filters early** - `?entity rdf:type {{type}}` narrows search space
5. **Leverage semantic variants** - If 0 results, auto-retry with `/kg-on` enabled

---

## Endpoint-Specific Notes

### LinkedData URIBurner (Primary)
- URL: `https://linkeddata.uriburner.com/sparql`
- Strengths: Rich FAQ, HowTo collections
- Response time: <500ms for typical queries

### Kingsley's Instance (Fallback 1)
- URL: `https://kingsley.idehen.net/sparql`
- Strengths: Experimental ontologies, inference-heavy queries
- Use when: Primary returns 0 results

### OpenLink Demo (Fallback 2)
- URL: `https://demo.openlinksw.com/sparql`
- Strengths: General-purpose RDF data
- Use when: Both primary and Kingsley return 0 results
