# Predefined SPARQL Query Templates

## Table of Contents
1. [Label Matching](#label-matching)
2. [Entity Exploration](#entity-exploration)
3. [Semantic Breakdown](#semantic-breakdown)
4. [Full Text Search](#full-text-search)
5. [Type Filtering](#type-filtering)
6. [Ranking & Scoring](#ranking--scoring)

## Label Matching

### Template 1: Basic Label Match (Virtuoso Full-Text)
Used for initial prompt-to-entity mapping. Matches prompt text against indexed labels with relevance scoring.

```sparql
SELECT DISTINCT ?entity ?value ?sc
WHERE {
  ?entity ?attribute ?value.
  ?value bif:contains("'{{prompt_text}}'") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT {{limit}}
```

**Parameters:**
- `{{prompt_text}}`: The user's prompt text (single terms combined with AND/OR)
- `{{limit}}`: Result limit (default: 10)

**Example:**
```sparql
SELECT DISTINCT ?entity ?value ?sc
WHERE {
  ?entity ?attribute ?value.
  ?value bif:contains("'semantic web' AND ontology") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT 10
```

### Template 2: Label Match with Property Focus
Restrict matching to specific properties (e.g., labels, titles).

```sparql
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
PREFIX skos: <http://www.w3.org/2004/02/skos/core#>

SELECT DISTINCT ?entity ?label ?sc
WHERE {
  ?entity ?labelProp ?label .
  ?label bif:contains("'{{prompt_text}}'") option (score ?sc) .
  FILTER (?labelProp IN (rdfs:label, skos:prefLabel, <http://schema.org/name>))
}
ORDER BY DESC(?sc)
LIMIT {{limit}}
```

---

## Entity Exploration

### Template 3: Entity Type Discovery
Identify available entity types and their prevalence in the graph.

```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX schema: <http://schema.org/>
PREFIX skos: <http://www.w3.org/2004/02/skos/core#>

SELECT ?type (COUNT(?entity) AS ?count) (SAMPLE(?entity) AS ?sample)
WHERE {
  ?entity rdf:type ?type .
  FILTER (?type IN (
    schema:FAQPage,
    schema:Question,
    schema:DefinedTermSet,
    skos:ConceptScheme,
    schema:HowTo
  ))
}
GROUP BY ?type
ORDER BY DESC(?count)
LIMIT 100
```

### Template 4: Entity Count by Namespace
Understand data distribution across ontologies/namespaces.

```sparql
SELECT ?namespace (COUNT(?entity) AS ?count)
WHERE {
  ?entity ?predicate ?object .
  BIND(STRBEFORE(STR(?entity), "#") AS ?ns1)
  BIND(IF(STRBEFORE(?ns1, "/") != "", STRBEFORE(?ns1, "/"), ?ns1) AS ?namespace)
}
GROUP BY ?namespace
ORDER BY DESC(?count)
LIMIT 20
```

---

## Semantic Breakdown

### Template 5: Query by Subject-Predicate-Object Components
After semantic breakdown of prompt into SPO components, query for matching patterns.

```sparql
SELECT DISTINCT ?subject ?predicate ?object
WHERE {
  ?subject ?predicate ?object .
  ?subject rdf:type {{subject_type}} .
  ?predicate rdfs:label {{predicate_label}} .
  ?object bif:contains("'{{object_value}}'") option (score ?sc)
}
ORDER BY DESC(?sc)
LIMIT {{limit}}
```

**Usage:** After decomposing "Show me questions about climate change":
- Subject: `?question` (type: schema:Question)
- Predicate: `about` / `schema:about`
- Object: "climate change"

### Template 6: Multi-Entity Pattern Matching
Match complex relationships between multiple entity types.

```sparql
PREFIX schema: <http://schema.org/>

SELECT ?entity1 ?relation ?entity2
WHERE {
  ?entity1 rdf:type {{entity_type1}} .
  ?entity1 ?relation ?entity2 .
  ?entity2 rdf:type {{entity_type2}} .
  FILTER (?relation IN ({{relation_list}}))
}
LIMIT {{limit}}
```

---

## Full Text Search

### Template 7: Multi-Field Full-Text Search (Virtuoso)
Search across multiple text fields with combined terms.

```sparql
PREFIX schema: <http://schema.org/>
PREFIX dct: <http://purl.org/dc/terms/>

SELECT ?entity ?title ?description ?sc
WHERE {
  ?entity a {{entity_type}} .
  OPTIONAL { ?entity schema:name ?title . }
  OPTIONAL { ?entity dct:description ?description . }
  (
    (?title bif:contains("'{{term1}}' AND '{{term2}}'") option (score ?sc1)) ||
    (?description bif:contains("'{{term1}}' AND '{{term2}}'") option (score ?sc2))
  )
  BIND(COALESCE(?sc1, ?sc2, 0) AS ?sc)
}
ORDER BY DESC(?sc)
LIMIT {{limit}}
```

### Template 8: Phrase Search with Fuzzy Matching
Find entities with similar text using fuzzy matching (for typos/variations).

```sparql
PREFIX schema: <http://schema.org/>

SELECT ?entity ?title (STRLEN(?title) AS ?titleLength)
WHERE {
  ?entity schema:name ?title .
  FILTER (bif:contains(?title, "{{phrase}}"))
}
ORDER BY ?titleLength
LIMIT {{limit}}
```

---

## Type Filtering

### Template 9: Scoped Type Query
Limit results to specific entity types with optional sub-filtering.

```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
PREFIX schema: <http://schema.org/>

SELECT ?entity ?label ?type
WHERE {
  ?entity rdf:type ?type .
  ?entity rdfs:label ?label .
  ?label bif:contains("'{{prompt_text}}'") option (score ?sc) .
  FILTER (?type IN ({{type_list}})) .
  OPTIONAL { ?entity schema:datePublished ?date . }
}
ORDER BY DESC(?sc)
LIMIT {{limit}}
```

### Template 10: Hierarchical Type Query with Inference
Query using owl:equivalentClass reasoning to match superclasses.

```sparql
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX owl: <http://www.w3.org/2002/07/owl#>
DEFINE input:inference 'urn:owl:equivalent:class:inference:rules'

SELECT ?entity ?actualType ?label
WHERE {
  ?entity rdf:type ?actualType .
  ?actualType owl:equivalentClass {{target_type}} .
  ?entity rdfs:label ?label .
}
LIMIT {{limit}}
```

---

## Ranking & Scoring

### Template 11: Combined Relevance Scoring
Score results by multiple factors: text match, date recency, popularity.

```sparql
PREFIX schema: <http://schema.org/>
PREFIX dct: <http://purl.org/dc/terms/>

SELECT ?entity ?label ?score
WHERE {
  ?entity rdfs:label ?label .
  ?label bif:contains("'{{prompt_text}}'") option (score ?textScore) .
  OPTIONAL { ?entity dct:modified ?modified . }
  OPTIONAL { ?entity schema:interactionCount ?views . }
  BIND(
    (?textScore * 0.7) +
    (IF(BOUND(?modified), (YEAR(NOW()) - YEAR(?modified)) * 0.1, 0)) +
    (IF(BOUND(?views), LOG(?views) * 0.2, 0)) AS ?score
  )
}
ORDER BY DESC(?score)
LIMIT {{limit}}
```

### Template 12: Similarity-Based Ranking with Semantic Variants
Retry with semantically similar terms when initial search yields no results.

```sparql
SELECT ?entity ?primaryMatch ?variantMatch ?combinedScore
WHERE {
  {
    ?entity rdfs:label ?label .
    ?label bif:contains("'{{primary_term}}'") option (score ?sc1) .
    BIND(?sc1 AS ?combinedScore)
    BIND("primary" AS ?primaryMatch)
  }
  UNION
  {
    ?entity rdfs:label ?label .
    ?label bif:contains("'{{semantic_variant1}}' OR '{{semantic_variant2}}'") option (score ?sc2) .
    BIND(?sc2 * 0.8 AS ?combinedScore)
    BIND("variant" AS ?variantMatch)
  }
}
ORDER BY DESC(?combinedScore)
LIMIT {{limit}}
```

---

## Reference Parameters

| Parameter | Example | Notes |
|-----------|---------|-------|
| `{{prompt_text}}` | "climate change" | User input; combine multiple terms with AND/OR |
| `{{limit}}` | 10 | Result limit; configurable via `/limit` command |
| `{{entity_type}}` | schema:Question | Scoped entity type; use namespace prefix |
| `{{type_list}}` | schema:FAQPage, schema:Question | Comma-separated types |
| `{{term1}}, {{term2}}` | "renewable", "energy" | Multiple search terms |
| `{{subject_type}}` | schema:Question | Type for subject after SPO decomposition |
| `{{relation_list}}` | schema:about, schema:mentions | Predicate IRIs to match |

## Execution Notes

- **Timeout:** 30 seconds per query (configurable)
- **Virtuoso specifics:** Use `bif:contains()` for full-text; combine terms in single clause
- **Fallback:** If primary endpoint times out, retry on fallback endpoints
- **IRI Encoding:** URL-encode all IRIs in results before creating describe links
