# Virtuoso SPIN Inference Rule Generator

A Claude Code skill for generating OpenLink Virtuoso SPIN-based custom inference rules from SPARQL queries.

## Overview

This skill automates the creation of SPIN (SPARQL Inferencing Notation) macro libraries for Virtuoso 8+. It converts analytical SPARQL queries into inference rules that can classify entities at query time without materializing derived data.

## What are SPIN Rules?

SPIN rules in Virtuoso enable runtime inference through macro expansion:
- Define derived classifications (e.g., "InactiveCustomer", "PremiumUser")
- Rules are stored as RDF in a named graph
- Activated per-query using `DEFINE input:macro-lib <URN>`
- No materialization required - inference happens at query time
- Based on W3C standard SPARQL CONSTRUCT queries

## Use Cases

- **Customer Segmentation**: Classify customers by recency, frequency, monetary value
- **Entity Classification**: Derive entity types based on properties and relationships
- **Temporal Analysis**: Categorize by time-based criteria (active, dormant, inactive)
- **Threshold-Based Rules**: Classify by counts, sums, or other aggregations
- **Multi-Criteria Classification**: Combine multiple conditions for complex rules

## Installation

### Via Claude Capabilities UI (Recommended)
1. Open Claude (web or desktop app)
2. Go to Settings → Capabilities
3. Click "Add Capability" or upload the skill
4. Upload `virtuoso-spin-rule-generator.zip`
5. The `/spin-rule-gen` command will be available immediately

### Manual Installation
Simply extract the zip file to access documentation and reference files:
```bash
unzip virtuoso-spin-rule-generator.zip
```

## Usage

### Invoke the Skill

```bash
/spin-rule-gen
```

or with Claude Code:

```
Use the spin-rule-gen skill to generate inference rules from my queries
```

### Input Methods

#### 1. From a Query File
```
Generate SPIN rules from the queries in customer-analysis.rq
```

#### 2. Direct SPARQL Query
```
Generate a SPIN rule for this query:

PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>
PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>

SELECT DISTINCT ?company ?mostRecentOrderDate
WHERE {
  {
    SELECT ?company (MAX(?orderDate) AS ?mostRecentOrderDate)
    WHERE {
      ?order a ecrm:Order ;
             ecrm:hasCompany ?company ;
             ecrm:orderDate ?orderDate .
      FILTER (datatype(?orderDate) IN (xsd:date, xsd:dateTime))
    }
    GROUP BY ?company
  }
  FILTER (?mostRecentOrderDate <= (NOW() - "P2Y"^^xsd:duration))
}
```

#### 3. Natural Language Description
```
Create a rule that classifies products as "BestSeller" if they have
been ordered more than 1000 times in the last year
```

## Output

The skill generates a complete `.sql` file with:

1. **Header**: Description and metadata
2. **Housekeeping**: Graph cleanup commands
3. **Preflight Controls**: Validation queries to verify data exists
4. **TTLP Section**: Turtle-based class definitions and SPIN rules
5. **Macro Generation**: Commands to compile and load the rules
6. **Inference Tests**: Positive and negative test queries

### Example Output Structure

```sql
-- Header with description
SPARQL CLEAR GRAPH <urn:...> ;
SPARQL DROP SPIN LIBRARY <urn:...> ;

-- Preflight queries
SPARQL SELECT ... LIMIT 10 ;

-- TTLP with class definitions and SPIN rules
TTLP('
  :DerivedClass a owl:Class ; rdfs:label "..." .
  source:Class spin:rule [ sp:text """CONSTRUCT { ... } WHERE { ... }""" ] .
', '', 'urn:...', 4096);

-- Macro generation
SELECT SPARQL_SPIN_GRAPH_TO_DEFSPIN('urn:...') ;
EXEC ( 'SPARQL ' || SPARQL_SPIN_GRAPH_TO_DEFSPIN('urn:...')) ;

-- Tests
SPARQL DEFINE input:macro-lib <urn:...> SELECT ... ;
```

## File Naming Convention

Generated files follow this pattern:
```
spin-rule-[domain]-[concept]-[version].sql
```

Examples:
- `spin-rule-customer-analytics-1.sql`
- `spin-rule-ecrm-premium-customers-1.sql`
- `spin-rule-product-classification-2.sql`

## Using Generated Rules in Virtuoso

### 1. Load the Rules
Execute the generated `.sql` file in Virtuoso iSQL:
```bash
isql localhost:1111 dba password < spin-rule-customer-analytics-1.sql
```

### 2. Query with Inference
```sparql
SPARQL
DEFINE input:macro-lib <urn:spin:ecrm:customer-analytics:lib>
PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>
PREFIX : <#>

SELECT ?company ?companyName
WHERE {
  ?company a ecrm:Company ;
           a :InactiveCustomer ;
           schema:name ?companyName .
}
```

### 3. Query Without Inference
```sparql
SPARQL
-- No DEFINE statement = no inference
PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>
PREFIX : <#>

SELECT ?company ?companyName
WHERE {
  ?company a ecrm:Company ;
           a :InactiveCustomer ;  -- Won't find any results
           schema:name ?companyName .
}
```

## Features

✅ Automatic class name generation from query patterns
✅ Comprehensive preflight validation queries
✅ Support for time-based classifications (durations)
✅ Support for aggregation-based rules (COUNT, MAX, MIN, SUM)
✅ Multi-criteria classifications
✅ Proper URN namespacing
✅ Complete test suite generation (positive and negative tests)
✅ Follows Virtuoso 8 best practices
✅ Clean, commented output

## Requirements

- OpenLink Virtuoso 8.0+
- Basic understanding of SPARQL
- RDF data loaded in Virtuoso

## Examples Included

See `REFERENCE.md` for:
- Minimal complete example
- Multiple rules pattern
- Common prefix declarations
- Duration syntax reference
- Testing patterns

## Troubleshooting

### "Preflight returns ZERO rows"
- Verify data exists in your Virtuoso instance
- Check that ontology URIs match your data
- Adjust time-based filters if using durations

### "Inference not working"
- Confirm `DEFINE input:macro-lib <URN>` is in your query
- Verify the URN matches what's in the SPIN library
- Check that macros were successfully generated (section 3)

### "Syntax error in TTLP"
- Ensure all triple quotes are balanced: `"""`
- Check semicolons between spin:rule blocks
- Verify period `.` after the last rule

## Advanced Usage

### Combining Multiple Libraries

You can activate multiple SPIN libraries in one query:
```sparql
DEFINE input:macro-lib <urn:spin:ecrm:customer-analytics:lib>
DEFINE input:macro-lib <urn:spin:ecrm:premium-customers:lib>
SELECT ...
```

### Conditional Inference

Use subqueries to apply inference selectively:
```sparql
SELECT * WHERE {
  {
    DEFINE input:macro-lib <urn:spin:...>
    SELECT ?s WHERE { ?s a :DerivedClass }
  }
  ?s :property ?value .
}
```

## Contributing

This skill is part of the OpenLink Virtuoso ecosystem. For issues or enhancements:
- Visit: https://github.com/openlink/virtuoso-opensource
- Forum: https://community.openlinksw.com

## License

MIT License - See LICENSE file

## Author

OpenLink Software
https://www.openlinksw.com

## Version

1.0.0 - Initial release
