# SPIN Rule Reference Pattern

This file provides a minimal complete example of a Virtuoso SPIN inference rule file.

## Minimal Complete Example

```sql
-------------------------------------------------------------------------------
-- SPIN-based Premium Customer Classification (Virtuoso 8)
-- Identifies customers with 50+ orders as premium customers
-------------------------------------------------------------------------------

------------------------------
-- 0. Housekeeping (rules only)
------------------------------
SPARQL CLEAR GRAPH <urn:spin:ecrm:premium:lib> ;
SPARQL DROP SPIN LIBRARY <urn:spin:ecrm:premium:lib> ;

-------------------------------------------------------------------------------
-- 1. PREFLIGHT CONTROLS
-------------------------------------------------------------------------------

SPARQL
PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>

SELECT DISTINCT ?company (COUNT(DISTINCT ?order) AS ?total)
WHERE {
  ?order a ecrm:Order ;
         ecrm:hasCompany ?company .
}
GROUP BY ?company
HAVING (COUNT(DISTINCT ?order) >= 50)
LIMIT 10 ;

-- STOP HERE if the above returns ZERO rows.

-------------------------------------------------------------------------------
-- 2. Load SPIN rules using TTLP()
-------------------------------------------------------------------------------

TTLP('
@prefix ecrm: <http://www.openlinksw.com/ontology/ecrm#> .
@prefix sp:   <http://spinrdf.org/sp#> .
@prefix spin: <http://spinrdf.org/spin#> .
@prefix rdf:  <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .
@prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .
@prefix owl:  <http://www.w3.org/2002/07/owl#> .
@prefix :     <#> .

## Derived Class
:PremiumCustomer
    a owl:Class ;
    rdfs:label "Premium Customer (50+ orders)" ;
    rdfs:comment "Company with 50 or more orders" .

## SPIN Rule
ecrm:Company
    a owl:Class ;
    rdfs:label "Company" ;
    spin:rule [
        a sp:Construct ;
        rdfs:label "Classify Premium Customer" ;
        rdfs:comment "Companies with 50+ orders" ;
        sp:text """
            PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>

            CONSTRUCT { ?company a <#PremiumCustomer> . }
            WHERE {
                {
                    SELECT ?company (COUNT(DISTINCT ?order) AS ?totalOrders)
                    WHERE {
                        ?order a ecrm:Order ;
                               ecrm:hasCompany ?company .
                    }
                    GROUP BY ?company
                }
                FILTER (?totalOrders >= 50)
            }
        """
    ] .
',
'', 'urn:spin:ecrm:premium:lib', 4096 );

-------------------------------------------------------------------------------
-- 3. Generate + execute SPIN macros
-------------------------------------------------------------------------------

SELECT SPARQL_SPIN_GRAPH_TO_DEFSPIN('urn:spin:ecrm:premium:lib');

EXEC ( 'SPARQL ' || SPARQL_SPIN_GRAPH_TO_DEFSPIN('urn:spin:ecrm:premium:lib'));

-------------------------------------------------------------------------------
-- 4. Inference Tests
-------------------------------------------------------------------------------

-- Positive (WITH inference — should return results)
SPARQL
DEFINE input:macro-lib <urn:spin:ecrm:premium:lib>
PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>
PREFIX : <#>

SELECT DISTINCT ?company
WHERE { ?company a ecrm:Company ; a :PremiumCustomer . }
LIMIT 10 ;

-- Negative (WITHOUT inference — should return ZERO)
SPARQL
-- DEFINE input:macro-lib <urn:spin:ecrm:premium:lib>
PREFIX ecrm: <http://www.openlinksw.com/ontology/ecrm#>
PREFIX : <#>

SELECT DISTINCT ?company
WHERE { ?company a ecrm:Company ; a :PremiumCustomer . }
LIMIT 10 ;
```

## Key Observations

1. **URN Consistency**: `urn:spin:ecrm:premium:lib` appears 6 times (housekeeping × 2, TTLP, macro gen × 2, tests × 2)

2. **Local Class Reference**: Inside CONSTRUCT, use `<#PremiumCustomer>` not `:PremiumCustomer`

3. **Triple Quotes**: SPARQL queries inside `sp:text` use triple quotes `"""`

4. **Turtle Syntax**: Everything between `TTLP('` and `', '', 'urn:...', 4096 );` is Turtle

5. **Semicolons vs Periods**:
   - Use `;` to continue adding properties to the same subject
   - Use `.` to end a subject's property list

6. **Aggregation Preservation**: The subquery with GROUP BY from preflight appears identically in the CONSTRUCT WHERE clause

## Multiple Rules Example

```turtle
ecrm:Company
    a owl:Class ;
    rdfs:label "Company" ;
    spin:rule [
        # First rule
        a sp:Construct ;
        rdfs:label "Rule 1" ;
        sp:text """..."""
    ] ;
    spin:rule [
        # Second rule
        a sp:Construct ;
        rdfs:label "Rule 2" ;
        sp:text """..."""
    ] ;
    spin:rule [
        # Third rule
        a sp:Construct ;
        rdfs:label "Rule 3" ;
        sp:text """..."""
    ] .
```

Note: Semicolon after each rule block except the last, which gets a period.

## Common Prefixes

Always include these in TTLP:
```turtle
@prefix sp:   <http://spinrdf.org/sp#> .
@prefix spin: <http://spinrdf.org/spin#> .
@prefix rdf:  <http://www.w3.org/1999/02/22-rdf-syntax-ns#> .
@prefix rdfs: <http://www.w3.org/2000/01/rdf-schema#> .
@prefix owl:  <http://www.w3.org/2002/07/owl#> .
@prefix :     <#> .
```

Plus your domain-specific prefixes:
```turtle
@prefix ecrm: <http://www.openlinksw.com/ontology/ecrm#> .
@prefix xsd:  <http://www.w3.org/2001/XMLSchema#> .
@prefix schema: <http://schema.org/> .
```

## Duration Syntax Reference

```sparql
# Years
"P1Y"^^xsd:duration   # 1 year
"P2Y"^^xsd:duration   # 2 years
"P5Y"^^xsd:duration   # 5 years

# Months
"P6M"^^xsd:duration   # 6 months
"P12M"^^xsd:duration  # 12 months

# Days
"P30D"^^xsd:duration  # 30 days
"P90D"^^xsd:duration  # 90 days
"P365D"^^xsd:duration # 365 days

# Combined
"P1Y6M"^^xsd:duration # 1 year 6 months
```

## Testing Pattern

Always test both modes:

```sparql
-- WITH inference (should find inferred triples)
DEFINE input:macro-lib <urn:spin:domain:concept:lib>
SELECT ?s WHERE { ?s a :DerivedClass . }

-- WITHOUT inference (should find nothing)
-- DEFINE input:macro-lib <urn:spin:domain:concept:lib>
SELECT ?s WHERE { ?s a :DerivedClass . }
```

The negative test proves the derived class doesn't exist in base data and is truly inferred.
